<?php

/**
 * This file is part of the Tracy (https://tracy.nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

namespace Tracy;

use Tracy\Dumper\Describer;
use Tracy\Dumper\Exposer;
use Tracy\Dumper\Renderer;

/**
 * Dumps a variable.
 */
final class Dumper
{
    const DEPTH = 'depth';

    const // how many nested levels of array/object properties display (defaults to 7)
        TRUNCATE = 'truncate';

    const // how truncate long strings? (defaults to 150)
        ITEMS = 'items';

    const // how many items in array/object display? (defaults to 100)
        COLLAPSE = 'collapse';

    const // collapse top array/object or how big are collapsed? (defaults to 14)
        COLLAPSE_COUNT = 'collapsecount';

    const // how big array/object are collapsed in non-lazy mode? (defaults to 7)
        LOCATION = 'location';

    const // show location string? (defaults to 0)
        OBJECT_EXPORTERS = 'exporters';

    const // custom exporters for objects (defaults to Dumper::$objectexporters)
        LAZY = 'lazy';

    const // lazy-loading via JavaScript? true=full, false=none, null=collapsed parts (defaults to null/false)
        LIVE = 'live';

    const // use static $liveSnapshot (used by Bar)
        SNAPSHOT = 'snapshot';

    const // array used for shared snapshot for lazy-loading via JavaScript
        DEBUGINFO = 'debuginfo';

    const // use magic method __debugInfo if exists (defaults to false)
        KEYS_TO_HIDE = 'keystohide';

    const // sensitive keys not displayed (defaults to [])
        SCRUBBER = 'scrubber';

    const // detects sensitive keys not to be displayed
        THEME = 'theme'; // color theme (defaults to light)

    const LOCATION_CLASS = 0b0001;

    const // shows where classes are defined
        LOCATION_SOURCE = 0b0011;

    const // additionally shows where dump was called
        LOCATION_LINK = self::LOCATION_SOURCE; // deprecated

    const HIDDEN_VALUE = Describer::HIDDEN_VALUE;

    /** @var Dumper\Value[] */
    public static $liveSnapshot = array();

    /** @var array */
    public static $terminalColors = array(
        'bool'      => '1;33',
        'null'      => '1;33',
        'number'    => '1;32',
        'string'    => '1;36',
        'array'     => '1;31',
        'public'    => '1;37',
        'protected' => '1;37',
        'private'   => '1;37',
        'dynamic'   => '1;37',
        'virtual'   => '1;37',
        'object'    => '1;31',
        'resource'  => '1;37',
        'indent'    => '1;30',
    );

    /** @var array */
    public static $resources = array(
        'stream'         => 'stream_get_meta_data',
        'stream-context' => 'stream_context_get_options',
        'curl'           => 'curl_getinfo',
    );

    /** @var array */
    public static $objectExporters = array(
        \Closure::class                => array(Exposer::class, 'exposeClosure'),
        \ArrayObject::class            => array(Exposer::class, 'exposeArrayObject'),
        \SplFileInfo::class            => array(Exposer::class, 'exposeSplFileInfo'),
        \SplObjectStorage::class       => array(Exposer::class, 'exposeSplObjectStorage'),
        \__PHP_Incomplete_Class::class => array(Exposer::class, 'exposePhpIncompleteClass'),
        \DOMNode::class                => array(Exposer::class, 'exposeDOMNode'),
        \DOMNodeList::class            => array(Exposer::class, 'exposeDOMNodeList'),
        \DOMNamedNodeMap::class        => array(Exposer::class, 'exposeDOMNodeList'),
    );

    /** @var Describer */
    private $describer;

    /** @var Renderer */
    private $renderer;

    private function __construct(array $options = array())
    {
        $location = $options[self::LOCATION] ?? 0;
        $location = $location === true ? ~0 : (int) $location;

        $describer = $this->describer = new Describer();
        $describer->maxDepth = $options[self::DEPTH] ?? $describer->maxDepth;
        $describer->maxLength = $options[self::TRUNCATE] ?? $describer->maxLength;
        $describer->maxItems = $options[self::ITEMS] ?? $describer->maxItems;
        $describer->debugInfo = $options[self::DEBUGINFO] ?? $describer->debugInfo;
        $describer->scrubber = $options[self::SCRUBBER] ?? $describer->scrubber;
        $describer->keysToHide = \array_flip(\array_map('strtolower', $options[self::KEYS_TO_HIDE] ?? array()));
        $describer->resourceExposers = ($options['resourceExporters'] ?? array()) + self::$resources;
        $describer->objectExposers = ($options[self::OBJECT_EXPORTERS] ?? array()) + self::$objectExporters;
        $describer->location = (bool) $location;
        if ($options[self::LIVE] ?? false)
        {
            $tmp = &self::$liveSnapshot;
        }
        elseif (isset($options[self::SNAPSHOT]))
        {
            $tmp = &$options[self::SNAPSHOT];
        }
        if (isset($tmp))
        {
            $tmp[0] = $tmp[0] ?? array();
            $tmp[1] = $tmp[1] ?? array();
            $describer->snapshot = &$tmp[0];
            $describer->references = &$tmp[1];
        }

        $renderer = $this->renderer = new Renderer();
        $renderer->collapseTop = $options[self::COLLAPSE] ?? $renderer->collapseTop;
        $renderer->collapseSub = $options[self::COLLAPSE_COUNT] ?? $renderer->collapseSub;
        $renderer->collectingMode = isset($options[self::SNAPSHOT]) || ! empty($options[self::LIVE]);
        $renderer->lazy = $renderer->collectingMode
            ? true
            : ($options[self::LAZY] ?? $renderer->lazy);
        $renderer->sourceLocation = ! (~$location & self::LOCATION_SOURCE);
        $renderer->classLocation = ! (~$location & self::LOCATION_CLASS);
        $renderer->theme = $options[self::THEME] ?? $renderer->theme;
    }

    /**
     * Dumps variable to the output.
     * @return mixed  variable
     */
    public static function dump($var, array $options = array())
    {
        if (PHP_SAPI === 'cli' || PHP_SAPI === 'phpdbg')
        {
            $useColors = self::$terminalColors && Helpers::detectColors();
            $dumper = new self($options);
            \fwrite(STDOUT, $dumper->asTerminal($var, $useColors ? self::$terminalColors : array()));
        }
        elseif (\preg_match('#^Content-Type: (?!text/html)#im', \implode("\n", \headers_list())))
        { // non-html
            echo self::toText($var, $options);
        }
        else
        { // html
            $options[self::LOCATION] = $options[self::LOCATION] ?? true;
            self::renderAssets();
            echo self::toHtml($var, $options);
        }

        return $var;
    }

    /**
     * Dumps variable to HTML.
     */
    public static function toHtml($var, array $options = array(), $key = null): string
    {
        return (new self($options))->asHtml($var, $key);
    }

    /**
     * Dumps variable to plain text.
     */
    public static function toText($var, array $options = array()): string
    {
        return (new self($options))->asTerminal($var);
    }

    /**
     * Dumps variable to x-terminal.
     */
    public static function toTerminal($var, array $options = array()): string
    {
        return (new self($options))->asTerminal($var, self::$terminalColors);
    }

    /**
     * Renders <script> & <style>
     */
    public static function renderAssets()
    {
        static $sent;
        if (Debugger::$productionMode === true || $sent)
        {
            return;
        }
        $sent = true;

        $nonce = Helpers::getNonce();
        $nonceAttr = $nonce ? ' nonce="' . Helpers::escapeHtml($nonce) . '"' : '';
        $s = \file_get_contents(__DIR__ . '/../Toggle/toggle.css')
            . \file_get_contents(__DIR__ . '/assets/dumper-light.css')
            . \file_get_contents(__DIR__ . '/assets/dumper-dark.css');
        echo "<style{$nonceAttr}>", \str_replace('</', '<\/', Helpers::minifyCss($s)), "</style>\n";

        if ( ! Debugger::isEnabled())
        {
            $s = '(function(){' . \file_get_contents(__DIR__ . '/../Toggle/toggle.js') . '})();'
                . '(function(){' . \file_get_contents(__DIR__ . '/../Dumper/assets/dumper.js') . '})();';
            echo "<script{$nonceAttr}>", \str_replace(array('<!--', '</s'), array('<\!--', '<\/s'), Helpers::minifyJs($s)), "</script>\n";
        }
    }

    public static function formatSnapshotAttribute(array &$snapshot): string
    {
        $res = '\'' . Renderer::jsonEncode($snapshot[0] ?? array()) . '\'';
        $snapshot = array();

        return $res;
    }

    /**
     * Dumps variable to HTML.
     */
    private function asHtml($var, $key = null): string
    {
        if ($key === null)
        {
            $model = $this->describer->describe($var);
        }
        else
        {
            $model = $this->describer->describe(array($key => $var));
            $model->value = $model->value[0][1];
        }

        return $this->renderer->renderAsHtml($model);
    }

    /**
     * Dumps variable to x-terminal.
     */
    private function asTerminal($var, array $colors = array()): string
    {
        $model = $this->describer->describe($var);

        return $this->renderer->renderAsText($model, $colors);
    }
}
